# autocorrect.tcl - 
#
# Copyright (c) 2005 Luciano Espirito Santo
# Copyright (c) 2005 Pat Thoyts
#
# Provide auto-correction for typing in the chat window entry field.
#
# $Id$

namespace eval autocorrect {

    variable scriptdir [file dirname [info script]]
    set sysenc [encoding system]
    encoding system utf-8
    msgcat::mcload [file join $scriptdir msgs]
    encoding system $sysenc

    custom::defgroup AutoCorrect \
        [msgcat::mc "AutoCorrect plugin options"] -group Chat

    custom::defvar options(use_autocorrect) 1 \
        [msgcat::mc "Enable AutoCorrect"] \
        -group AutoCorrect -type boolean

    custom::defvar options(corrections_file) \
        [file join ~ .tkabber autocorrect.txt] \
        [msgcat::mc "Path to corrections file"] \
        -group AutoCorrect -type string

    hook::add finload_hook [namespace current]::init_plugin
    hook::add open_chat_post_hook [namespace current]::init_chat
}

# Add our plugin to the Services>Plugins menu.
#
proc autocorrect::init_plugin {} {
    catch {
        set menu [.mainframe getmenu plugins]
        $menu add cascade -label [msgcat::mc "AutoCorrect"] \
            -menu [set acm [menu $menu.autocorrect -tearoff 0]]
        $acm add checkbutton \
            -label [msgcat::mc "AutoCorrect"] \
            -underline 0 \
            -variable [namespace current]::options(use_autocorrect)
        $acm add command \
            -label [msgcat::mc "Edit corrections"] \
            -underline 0 \
            -command [namespace origin edit_corrections]
    }

    image create photo [namespace current]::img-file -data {
        R0lGODlhDgAQAOcTAMbO9+fn///OAJwAAN7e53OEvf//nP8AY97e3s7OnJyc
        Y73G987O9///AHuUvYSUvd7e/9be5wCEAFJjlGt7tQD/AM7OznuMrXuMtb3G
        1vf398bO76W157W93nuMvff3/5xjAOfn5+/v/4SEhP///wAAAP///ycnJygo
        KCkpKSoqKisrKywsLC0tLS4uLi8vLzAwMDExMTIyMjMzMzQ0NDU1NTY2Njc3
        Nzg4ODk5OTo6Ojs7Ozw8PD09PT4+Pj8/P0BAQEFBQUJCQkNDQ0REREVFRUZG
        RkdHR0hISElJSUpKSktLS0xMTE1NTU5OTk9PT1BQUFFRUVJSUlNTU1RUVFVV
        VVZWVldXV1hYWFlZWVpaWltbW1xcXF1dXV5eXl9fX2BgYGFhYWJiYmNjY2Rk
        ZGVlZWZmZmdnZ2hoaGlpaWpqamtra2xsbG1tbW5ubm9vb3BwcHFxcXJycnNz
        c3R0dHV1dXZ2dnd3d3h4eHl5eXp6ent7e3x8fH19fX5+fn9/f4CAgIGBgYKC
        goODg4SEhIWFhYaGhoeHh4iIiImJiYqKiouLi4yMjI2NjY6Ojo+Pj5CQkJGR
        kZKSkpOTk5SUlJWVlZaWlpeXl5iYmJmZmZqampubm5ycnJ2dnZ6enp+fn6Cg
        oKGhoaKioqOjo6SkpKWlpaampqenp6ioqKmpqaqqqqurq6ysrK2tra6urq+v
        r7CwsLGxsbKysrOzs7S0tLW1tba2tre3t7i4uLm5ubq6uru7u7y8vL29vb6+
        vr+/v8DAwMHBwcLCwsPDw8TExMXFxcbGxsfHx8jIyMnJycrKysvLy8zMzM3N
        zc7Ozs/Pz9DQ0NHR0dLS0tPT09TU1NXV1dbW1tfX19jY2NnZ2dra2tvb29zc
        3N3d3d7e3t/f3+Dg4OHh4eLi4uPj4+Tk5OXl5ebm5ufn5+jo6Onp6erq6uvr
        6+zs7O3t7e7u7u/v7/Dw8PHx8fLy8vPz8/T09PX19fb29vf39/j4+Pn5+fr6
        +vv7+/z8/P39/f7+/v///yH+FUNyZWF0ZWQgd2l0aCBUaGUgR0lNUAAh+QQB
        CgD/ACwAAAAADgAQAAAIeAD/CRxIsKDBgwgNjljIsGGJgSMsIAhBMUSEDh0e
        ChyBQAOJjx8+kMgIMcTHkx8faPw3wiTIkCRUljz5QYSImCtbgrQpAoDMjSFE
        1rTpMycBlEQdGLUZoGmABR5yZtjAAIDVBRwK5LyAwYNXDwUoTFhZoqzZsysD
        AgA7
    }

    image create photo [namespace current]::img-modified -data {
        R0lGODlhDgAQAKUTAMbO9+fn///OAJwAAN7e53OEvf//nP8AY97e3s7OnJyc
        Y73G987O9///AHuUvYSUvd7e/9be5wCEAFJjlGt7tQD/AM7OznuMrXuMtb3G
        1vf398bO76W157W93nuMvff3/5xjAOfn5+/v/4SEhP///wAAAFJjlFJjlFJj
        lFJjlFJjlFJjlFJjlFJjlFJjlFJjlFJjlFJjlFJjlFJjlFJjlFJjlFJjlFJj
        lFJjlFJjlFJjlFJjlFJjlFJjlFJjlFJjlCH5BAEKAD8ALAAAAAAOABAAAAZ0
        wJ9wSPwNBsXk4HAoJYmgiqQ0qlqtIIPAObIgQuBQpANqgLgIDWn9EWW3wlFo
        TXcLzvE5+6NI4PN0bQokCU6AJG0iJYuGP3IfiCKKJQ+NIwRXigAOlgSSAaAQ
        Cx6WGRsMAKkLHAWWFxgesR4FFBONjLi4Q0EAOw==
    }

}

proc autocorrect::init_chat {chatid type} {
    variable options
    if {$options(use_autocorrect)} {
        loadAC
        
        set cw [chat::winid $chatid]    
        foreach i  { space  .  ,  :  ;  ?  !  \"  '  =  (  )  [  ] }  	{
            bind $cw.input <Key-$i> [list [namespace origin autocorrect] %W]
        }
        bind $cw.input <Control_L><F11> [list [namespace origin loadAC]]
    }
}

proc autocorrect::loadAC  {}  {
    variable options
    variable ACLIST
    array unset ACLIST
    if {[file exists $options(corrections_file)]} {
        set f [open $options(corrections_file) r]
        
        while {-1 != [gets $f line]} {
            if {!([regexp {^#} $line] || [string length $line] < 1)} {
                if {[regexp {^\s*(\S+)\s*=\s*(\S+)} $line -> a b]} {
                    set ACLIST($a) $b
                }
            }
        }
        
        close $f
    } else {
        array set ACLIST {teh the tcl Tcl tk Tk tkaber tkabber worng wrong}
    }
}

proc autocorrect::autocorrect {w} {
    variable ACLIST
    set _trail [$w get 1.0 insert]
    set _typeString "\[^\"\'(\\\[ \]+"
    
    for {set _c 1}  {$_c <= 5}  {incr _c}	{
        
        set _lastWord ""
        regexp -line  "($_typeString)\$"  $_trail  =>  _lastWord
        set _lastWordWipeSize [string length $_lastWord]
        
        if  {[info exists ACLIST($_lastWord)]}	{
            $w delete "insert -$_lastWordWipeSize c" insert
            $w insert insert "$ACLIST($_lastWord)"
            break
        }
        set _lastWord [string tolower $_lastWord]
        if  {[info exists ACLIST($_lastWord)]}  {
            $w delete "insert -$_lastWordWipeSize c" insert
            $w insert insert [string toupper $ACLIST($_lastWord)]
            break
        }
        set _typeString  "$_typeString \[^ \]+"
    }
}

proc autocorrect::edit_corrections {} {
    variable options
    global font

    set w .autocorrections
    if {[winfo exists $w]} { return }

    add_win $w -title [msgcat::mc "Auto-corrections"] \
        -tabtitle [msgcat::mc "Auto-corrections"] \
        -class Customize

    set sw [ScrolledWindow $w.sw]
    set t [text $w.txt]
    $sw setwidget $t

    if {[file exists $options(corrections_file)]} {
        set f [open $options(corrections_file) r]
        $t insert end [read $f]
        close $f
    }
    
    frame $w.f
    label $w.f.l -anchor w \
        -text [msgcat::mc "Editing %s" $options(corrections_file)]
    label $w.f.mod -image [namespace current]::img-file
    button $w.f.save -text [msgcat::mc "Save"] \
        -command [list [namespace origin save_corrections] $w $t]
    pack $w.f.save $w.f.mod -side right
    pack $w.f.l -side left -expand 1 -fill x
    pack $w.f -side top -fill x
    pack $w.sw -side top -fill both -expand 1

    $t edit reset
    $t edit modified 0
    bind $t <<Modified>> [list [namespace origin modified] $w $t]

    update idletasks
    focus $t
}

proc autocorrect::save_corrections {w tw} {
    variable options
    variable ACLIST
    global status
    set valid 1
    array set New {}
    set ::TW $tw
    eval [linsert [$tw tag names] 0 $tw tag delete]
    $tw tag configure ERR -background pink
    set max [lindex [split [$tw index end] .] 0]
    for {set n 1} {$n < $max} {incr n} {
        set line [$tw get $n.0 $n.end]
        if {!([regexp {^\#} $line] || [string length $line] < 1)} {
            if {[regexp {^\s*(\S+)\s*=\s*(\S+)} $line -> a b]} {
                set New($a) $b
            } else {
                debugmsg autocorrect "autocorrect error: '$line' $n.0"
                set valid 0
                $tw tag add ERR $n.0 $n.end
            }
        }
    }

    if {$valid} {
        variable ACLIST
        array unset ACLIST
        array set ACLIST [array get New]
        set f [open $options(corrections_file) w]
        puts $f [$tw get 0.0 end]
        close $f
        $tw edit modified 0
        set status [msgcat::mc "File saved"]
    } else {
        set status [msgcat::mc "File not saved due to format errors"]
    }
}

proc autocorrect::modified {w tw} {
    if {[$tw edit modified]} {
        $w.f.mod configure -image [namespace current]::img-modified
    } else {
        $w.f.mod configure -image [namespace current]::img-file
    }
}