static char _service_c[] =
"$Header: /cvs/tcl/tclsvc/service.c,v 1.1 1999/05/16 04:24:29 matt Exp $";
/*
 * Copyright (C) 1997-1999 Sensus Consulting Ltd.
 * Matt Newman <matt@sensus.org>
 */
/*
 * Derived from:
 *
 * THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF
 * ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
 * PARTICULAR PURPOSE.
 *
 * Copyright 1993 - 1998 Microsoft Corporation.
 *
 *  MODULE:   service.c
 *
 *  PURPOSE:  Implements functions required by all services
 *            windows.
 *
 *  FUNCTIONS:
 *    main(int argc, char **argv);
 *    service_ctrl(DWORD dwCtrlCode);
 *    service_main(DWORD dwArgc, LPTSTR *lpszArgv);
 *    CmdInstallService();
 *    CmdRemoveService();
 *    CmdDebugService(int argc, char **argv);
 *    ControlHandler ( DWORD dwCtrlType );
 *    GetLastErrorText( LPTSTR lpszBuf, DWORD dwSize );
 *
 *  COMMENTS:
 *
 *  AUTHOR: Craig Link - Microsoft Developer Support
 */


#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <process.h>
#include <tchar.h>
#include <locale.h>

/* list of service dependencies - "dep1\0dep2\0\0" */
#define SZDEPENDENCIES       ""

VOID ServiceStart(DWORD dwArgc, LPTSTR *lpszArgv);
VOID ServiceStop();
BOOL ReportStatusToSCMgr(DWORD dwCurrentState, DWORD dwWin32ExitCode, DWORD dwWaitHint);
void AddToMessageLog(int type, LPTSTR lpszMsg);

/* internal variables */
SERVICE_STATUS          ssStatus;       // current status of the service
SERVICE_STATUS_HANDLE   sshStatusHandle;
DWORD                   dwErr = 0;
BOOL                    bDebug = FALSE;
TCHAR                   szErr[256];
TCHAR                   szService[256];
TCHAR                   szDisplayName[256];
TCHAR                   szDepend[256] = "";

/* internal function prototypes */
VOID WINAPI service_ctrl(DWORD dwCtrlCode);
VOID WINAPI service_main(DWORD dwArgc, LPTSTR *lpszArgv);
VOID CmdInstallService(DWORD dwArgc, LPTSTR *lpszArgv);
VOID CmdRemoveService();
VOID CmdDebugService(int argc, char **argv);
BOOL WINAPI ControlHandler ( DWORD dwCtrlType );
LPTSTR GetLastErrorText( LPTSTR lpszBuf, DWORD dwSize );
static void		setargv	(int *argcPtr, char ***argvPtr);

/*
 *  FUNCTION: main
 *
 *  PURPOSE: entrypoint for service
 *
 *  PARAMETERS:
 *    argc - number of command line arguments
 *    argv - array of command line arguments
 *
 *  RETURN VALUE:
 *    none
 *
 *  COMMENTS:
 *    main() either performs the command line task, or
 *    call StartServiceCtrlDispatcher to register the
 *    main service thread.  When the this call returns,
 *    the service has stopped, so exit.
 */
void _cdecl main(int argc, char **argv)
{
    SERVICE_TABLE_ENTRY dispatchTable[2];
    char *p;
    char buffer[MAX_PATH];

    /*
     * Set up the default locale to be standard "C" locale so parsing
     * is performed correctly.
     */
    setlocale(LC_ALL, "C");

    setargv(&argc, &argv);

    /*
     * Replace argv[0] with full pathname of executable, and forward
     * slashes substituted for backslashes.
     */
    GetModuleFileName(NULL, buffer, sizeof(buffer));
    argv[0] = buffer;
    for (p = buffer; *p != '\0'; p++) {
	if (*p == '\\') {
	    *p = '/';
	}
    }
    /* Compute name of service from executable name */
    if ((p = strrchr( argv[0], '/')) != NULL) {
	argv[0] = p+1;
    }
    if ((p = strchr( argv[0], '.')) != NULL) {
	*p = '\0';
    }
    strcpy( szService, argv[0]);
    strcpy( szDisplayName, szService);

    dispatchTable[0].lpServiceName = szService;
    dispatchTable[0].lpServiceProc = (LPSERVICE_MAIN_FUNCTION)service_main;
    dispatchTable[1].lpServiceName = NULL;
    dispatchTable[1].lpServiceProc = NULL;

    if ( (argc > 1) &&
         ((*argv[1] == '-') || (*argv[1] == '/')) ) {
        if ( _stricmp( "install", argv[1]+1 ) == 0 ) {
            CmdInstallService(argc-1, &argv[1]);
        }
        else if ( _stricmp( "remove", argv[1]+1 ) == 0 ) {
            CmdRemoveService();
        }
        else if ( _stricmp( "debug", argv[1]+1 ) == 0 ) {
	    argc -= 1;
	    argv = &argv[1];
            bDebug = TRUE;
            CmdDebugService(argc, argv);
        }
        else {
	    fprintf( stderr, "Usage: %s -install ?<display name>? ?depend,depend?| -remove | -debug <args>\n", argv[0]);
	    exit(1);
        }
        exit(0);
    }
    // if it doesn't match any of the above parameters
    // the service control manager may be starting the service
    // so we must call StartServiceCtrlDispatcher
    dispatch:
    // this is just to be friendly
    printf( "\nStartServiceCtrlDispatcher being called.\n" );
    printf( "This may take several seconds.  Please wait.\n" );

    if (!StartServiceCtrlDispatcher(dispatchTable))
	AddToMessageLog(EVENTLOG_ERROR_TYPE,TEXT("StartServiceCtrlDispatcher failed."));
}

/*
 *  FUNCTION: service_main
 *
 *  PURPOSE: To perform actual initialization of the service
 *
 *  PARAMETERS:
 *    dwArgc   - number of command line arguments
 *    lpszArgv - array of command line arguments
 *
 *  RETURN VALUE:
 *    none
 *
 *  COMMENTS:
 *    This routine performs the service initialization and then calls
 *    the user defined ServiceStart() routine to perform majority
 *    of the work.
 */
void WINAPI service_main(DWORD dwArgc, LPTSTR *lpszArgv)
{

    // register our service control handler:
    //
    sshStatusHandle = RegisterServiceCtrlHandler( TEXT(szService), service_ctrl);

    if (!sshStatusHandle)
        goto cleanup;

    // SERVICE_STATUS members that don't change in example
    //
    ssStatus.dwServiceType = SERVICE_WIN32_OWN_PROCESS;
    ssStatus.dwServiceSpecificExitCode = 0;


    // report the status to the service control manager.
    //
    if (!ReportStatusToSCMgr(
        SERVICE_START_PENDING, // service state
        NO_ERROR,              // exit code
        3000))                 // wait hint
        goto cleanup;


    ServiceStart( dwArgc, lpszArgv );

cleanup:

    // try to report the stopped status to the service control manager.
    //
    if (sshStatusHandle)
        (VOID)ReportStatusToSCMgr(
                            SERVICE_STOPPED,
                            dwErr,
                            0);

    return;
}

/*
 *  FUNCTION: service_ctrl
 *
 *  PURPOSE: This function is called by the SCM whenever
 *           ControlService() is called on this service.
 *
 *  PARAMETERS:
 *    dwCtrlCode - type of control requested
 *
 *  RETURN VALUE:
 *    none
 *
 *  COMMENTS:
 */
VOID WINAPI service_ctrl(DWORD dwCtrlCode)
{
    // Handle the requested control code.
    //
    switch(dwCtrlCode)
    {
        // Stop the service.
        //
        // SERVICE_STOP_PENDING should be reported before
        // setting the Stop Event - hServerStopEvent - in
        // ServiceStop().  This avoids a race condition
        // which may result in a 1053 - The Service did not respond...
        // error.
        case SERVICE_CONTROL_STOP:
            ReportStatusToSCMgr(SERVICE_STOP_PENDING, NO_ERROR, 0);
            ServiceStop();
            return;

        // Update the service status.
        //
        case SERVICE_CONTROL_INTERROGATE:
            break;

        // invalid control code
        //
        default:
            break;

    }

    ReportStatusToSCMgr(ssStatus.dwCurrentState, NO_ERROR, 0);
}

/*
 *  FUNCTION: ReportStatusToSCMgr()
 *
 *  PURPOSE: Sets the current status of the service and
 *           reports it to the Service Control Manager
 *
 *  PARAMETERS:
 *    dwCurrentState - the state of the service
 *    dwWin32ExitCode - error code to report
 *    dwWaitHint - worst case estimate to next checkpoint
 *
 *  RETURN VALUE:
 *    TRUE  - success
 *    FALSE - failure
 *
 *  COMMENTS:
 */
BOOL ReportStatusToSCMgr(DWORD dwCurrentState,
                         DWORD dwWin32ExitCode,
                         DWORD dwWaitHint)
{
    static DWORD dwCheckPoint = 1;
    BOOL fResult = TRUE;


    if ( !bDebug ) // when debugging we don't report to the SCM
    {
        if (dwCurrentState == SERVICE_START_PENDING)
            ssStatus.dwControlsAccepted = 0;
        else
            ssStatus.dwControlsAccepted = SERVICE_ACCEPT_STOP;

        ssStatus.dwCurrentState = dwCurrentState;
        ssStatus.dwWin32ExitCode = dwWin32ExitCode;
        ssStatus.dwWaitHint = dwWaitHint;

        if ( ( dwCurrentState == SERVICE_RUNNING ) ||
             ( dwCurrentState == SERVICE_STOPPED ) )
            ssStatus.dwCheckPoint = 0;
        else
            ssStatus.dwCheckPoint = dwCheckPoint++;


        // Report the status of the service to the service control manager.
        //
        if (!(fResult = SetServiceStatus( sshStatusHandle, &ssStatus))) {
            AddToMessageLog(EVENTLOG_ERROR_TYPE,TEXT("SetServiceStatus"));
        }
    }
    return fResult;
}

/*
 *  FUNCTION: AddToMessageLog(LPTSTR lpszMsg)
 *
 *  PURPOSE: Allows any thread to log an error message
 *
 *  PARAMETERS:
 *    lpszMsg - text for message
 *
 *  RETURN VALUE:
 *    none
 *
 *  COMMENTS:
 */
VOID AddToMessageLog(int type, LPTSTR lpszMsg)
{
    TCHAR   szMsg[256];
    HANDLE  hEventSource;
    LPTSTR  lpszStrings[2];


    dwErr = GetLastError();

    // Use event logging to log the error.
    //
    hEventSource = RegisterEventSource(NULL, TEXT(szService));

    if (type == EVENTLOG_ERROR_TYPE) {
	_stprintf(szMsg, TEXT("%s error: %d"), TEXT(szService), dwErr);
    } else {
	_stprintf(szMsg, TEXT("%s:"), TEXT(szService));
    }
    lpszStrings[0] = TEXT(szService);
    lpszStrings[0] = lpszMsg;

    if (hEventSource == NULL) {
	    return;
    }
    ReportEvent(hEventSource, // handle of event source
	    (short)type,		  // event type
	    0,                    // event category
	    1,                    // event ID
	    NULL,                 // current user's SID
	    1,                    // strings in lpszStrings
	    0,                    // no bytes of raw data
	    lpszStrings,          // array of error strings
	    NULL);                // no raw data

    (VOID) DeregisterEventSource(hEventSource);
}

/*
 *  The following code handles service installation and removal
 *
 *  FUNCTION: CmdInstallService()
 *
 *  PURPOSE: Installs the service
 *
 *  PARAMETERS:
 *    none
 *
 *  RETURN VALUE:
 *    none
 *
 *  COMMENTS:
 */
VOID CmdInstallService(DWORD dwArgc, LPTSTR *lpszArgv)
{
    SC_HANDLE   schService;
    SC_HANDLE   schSCManager;
    TCHAR szPath[512], *p;

    if (dwArgc > 1) {
	strcpy( szDisplayName, lpszArgv[1]);
    }
    if (dwArgc > 2) {
	strcpy( szDepend, lpszArgv[2]);
	for (p=szDepend;*p;p++) {
	    if (*p == ',')
		*p = '\0';
	}
	*(++p) = '\0';
    }
    if ( GetModuleFileName( NULL, szPath, 512 ) == 0 )
    {
        _tprintf(TEXT("Unable to install %s - %s\n"), TEXT(szDisplayName), GetLastErrorText(szErr, 256));
        return;
    }

    schSCManager = OpenSCManager(
                        NULL,                   // machine (NULL == local)
                        NULL,                   // database (NULL == default)
                        SC_MANAGER_ALL_ACCESS   // access required
                        );
    if ( schSCManager )
    {
        schService = CreateService(
            schSCManager,               // SCManager database
            TEXT(szService),		// name of service
            TEXT(szDisplayName),	// name to display
            SERVICE_ALL_ACCESS,         // desired access
            SERVICE_WIN32_OWN_PROCESS,  // service type
            SERVICE_AUTO_START,		// start type
            SERVICE_ERROR_NORMAL,       // error control type
            szPath,                     // service's binary
            NULL,                       // no load ordering group
            NULL,                       // no tag identifier
            szDepend,       		// dependencies
            NULL,                       // LocalSystem account
            NULL);                      // no password

        if ( schService )
        {
            _tprintf(TEXT("%s installed.\n"), TEXT(szDisplayName) );
            CloseServiceHandle(schService);
        }
        else
        {
            _tprintf(TEXT("CreateService failed - %s\n"), GetLastErrorText(szErr, 256));
        }

        CloseServiceHandle(schSCManager);
    }
    else
        _tprintf(TEXT("OpenSCManager failed - %s\n"), GetLastErrorText(szErr,256));
}

/*
 *  FUNCTION: CmdRemoveService()
 *
 *  PURPOSE: Stops and removes the service
 *
 *  PARAMETERS:
 *    none
 *
 *  RETURN VALUE:
 *    none
 *
 *  COMMENTS:
 */
void CmdRemoveService()
{
    SC_HANDLE   schService;
    SC_HANDLE   schSCManager;

    schSCManager = OpenSCManager(
                        NULL,                   // machine (NULL == local)
                        NULL,                   // database (NULL == default)
                        SC_MANAGER_ALL_ACCESS   // access required
                        );
    if ( schSCManager )
    {
        schService = OpenService(schSCManager, TEXT(szService), SERVICE_ALL_ACCESS);

        if (schService)
        {
            // try to stop the service
            if ( ControlService( schService, SERVICE_CONTROL_STOP, &ssStatus ) )
            {
                _tprintf(TEXT("Stopping %s."), TEXT(szDisplayName));
                Sleep( 1000 );

                while( QueryServiceStatus( schService, &ssStatus ) )
                {
                    if ( ssStatus.dwCurrentState == SERVICE_STOP_PENDING )
                    {
                        _tprintf(TEXT("."));
                        Sleep( 1000 );
                    }
                    else
                        break;
                }

                if ( ssStatus.dwCurrentState == SERVICE_STOPPED )
                    _tprintf(TEXT("\n%s stopped.\n"), TEXT(szDisplayName) );
                else
                    _tprintf(TEXT("\n%s failed to stop.\n"), TEXT(szDisplayName) );

            }

            // now remove the service
            if( DeleteService(schService) )
                _tprintf(TEXT("%s removed.\n"), TEXT(szDisplayName) );
            else
                _tprintf(TEXT("DeleteService failed - %s\n"), GetLastErrorText(szErr,256));


            CloseServiceHandle(schService);
        }
        else
            _tprintf(TEXT("OpenService failed - %s\n"), GetLastErrorText(szErr,256));

        CloseServiceHandle(schSCManager);
    }
    else
        _tprintf(TEXT("OpenSCManager failed - %s\n"), GetLastErrorText(szErr,256));
}




/*
 *  The following code is for running the service as a console app
 *
 *  FUNCTION: CmdDebugService(int argc, char ** argv)
 *
 *  PURPOSE: Runs the service as a console application
 *
 *  PARAMETERS:
 *    argc - number of command line arguments
 *    argv - array of command line arguments
 *
 *  RETURN VALUE:
 *    none
 *
 *  COMMENTS:
 */
void CmdDebugService(int argc, char ** argv)
{
    DWORD dwArgc;
    LPTSTR *lpszArgv;

#ifdef UNICODE
    lpszArgv = CommandLineToArgvW(GetCommandLineW(), &(dwArgc) );
#else
    dwArgc   = (DWORD) argc;
    lpszArgv = argv;
#endif

    _tprintf(TEXT("Debugging %s.\n"), TEXT(szService));

    SetConsoleCtrlHandler( ControlHandler, TRUE );

    ServiceStart( dwArgc, lpszArgv );
}


/*
 *  FUNCTION: ControlHandler ( DWORD dwCtrlType )
 *
 *  PURPOSE: Handled console control events
 *
 *  PARAMETERS:
 *    dwCtrlType - type of control event
 *
 *  RETURN VALUE:
 *    True - handled
 *    False - unhandled
 *
 *  COMMENTS:
 */
BOOL WINAPI ControlHandler ( DWORD dwCtrlType )
{
    switch( dwCtrlType )
    {
        case CTRL_BREAK_EVENT:  // use Ctrl+C or Ctrl+Break to simulate
        case CTRL_C_EVENT:      // SERVICE_CONTROL_STOP in debug mode
            _tprintf(TEXT("Stopping %s.\n"), TEXT(szDisplayName));
            ServiceStop();
            return TRUE;
            break;

    }
    return FALSE;
}

/*
 *  FUNCTION: GetLastErrorText
 *
 *  PURPOSE: copies error message text to string
 *
 *  PARAMETERS:
 *    lpszBuf - destination buffer
 *    dwSize - size of buffer
 *
 *  RETURN VALUE:
 *    destination buffer
 *
 *  COMMENTS:
 */
LPTSTR GetLastErrorText( LPTSTR lpszBuf, DWORD dwSize )
{
    DWORD dwRet;
    LPTSTR lpszTemp = NULL;

    dwRet = FormatMessage( FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM |FORMAT_MESSAGE_ARGUMENT_ARRAY,
                           NULL,
                           GetLastError(),
                           LANG_NEUTRAL,
                           (LPTSTR)&lpszTemp,
                           0,
                           NULL );

    // supplied buffer is not long enough
    if ( !dwRet || ( (long)dwSize < (long)dwRet+14 ) )
        lpszBuf[0] = TEXT('\0');
    else
    {
        lpszTemp[lstrlen(lpszTemp)-2] = TEXT('\0');  //remove cr and newline character
        _stprintf( lpszBuf, TEXT("%s (0x%x)"), lpszTemp, GetLastError() );
    }

    if ( lpszTemp )
        LocalFree((HLOCAL) lpszTemp );

    return lpszBuf;
}

/*
 *-------------------------------------------------------------------------
 *
 * setargv --
 *
 *	Parse the Windows command line string into argc/argv.  Done here
 *	because we don't trust the builtin argument parser in crt0.  
 *	Windows applications are responsible for breaking their command
 *	line into arguments.
 *
 *	2N backslashes + quote -> N backslashes + begin quoted string
 *	2N + 1 backslashes + quote -> literal
 *	N backslashes + non-quote -> literal
 *	quote + quote in a quoted string -> single quote
 *	quote + quote not in quoted string -> empty string
 *	quote -> begin quoted string
 *
 * Results:
 *	Fills argcPtr with the number of arguments and argvPtr with the
 *	array of arguments.
 *
 * Side effects:
 *	Memory allocated.
 *
 *--------------------------------------------------------------------------
 */

static void
setargv(argcPtr, argvPtr)
    int *argcPtr;		/* Filled with number of argument strings. */
    char ***argvPtr;		/* Filled with argument strings (malloc'd). */
{
    char *cmdLine, *p, *arg, *argSpace;
    char **argv;
    int argc, size, inquote, copy, slashes;
    
    cmdLine = GetCommandLine();

    /*
     * Precompute an overly pessimistic guess at the number of arguments
     * in the command line by counting non-space spans.
     */

    size = 2;
    for (p = cmdLine; *p != '\0'; p++) {
	if (isspace(*p)) {
	    size++;
	    while (isspace(*p)) {
		p++;
	    }
	    if (*p == '\0') {
		break;
	    }
	}
    }
    argSpace = (char *) malloc((unsigned) (size * sizeof(char *) 
	    + strlen(cmdLine) + 1));
    argv = (char **) argSpace;
    argSpace += size * sizeof(char *);
    size--;

    p = cmdLine;
    for (argc = 0; argc < size; argc++) {
	argv[argc] = arg = argSpace;
	while (isspace(*p)) {
	    p++;
	}
	if (*p == '\0') {
	    break;
	}

	inquote = 0;
	slashes = 0;
	while (1) {
	    copy = 1;
	    while (*p == '\\') {
		slashes++;
		p++;
	    }
	    if (*p == '"') {
		if ((slashes & 1) == 0) {
		    copy = 0;
		    if ((inquote) && (p[1] == '"')) {
			p++;
			copy = 1;
		    } else {
			inquote = !inquote;
		    }
                }
                slashes >>= 1;
            }

            while (slashes) {
		*arg = '\\';
		arg++;
		slashes--;
	    }

	    if ((*p == '\0') || (!inquote && isspace(*p))) {
		break;
	    }
	    if (copy != 0) {
		*arg = *p;
		arg++;
	    }
	    p++;
        }
	*arg = '\0';
	argSpace = arg + 1;
    }
    argv[argc] = NULL;

    *argcPtr = argc;
    *argvPtr = argv;
}
