# tclxmppdd.tcl - Copyright (C) 2004 Pat Thoyts <patthoyts@users.sf.net>
#
# An implementation of an Jabber/XMPP server
# See RFC3920  http://www.xmpp.org/specs/rfc3920.html
#
# -------------------------------------------------------------------------
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
# -------------------------------------------------------------------------
#
# $Id$

package require tdom
#package require Mk4tcl
package require logger;                 # tcllib
package require uuid;                   # tcllib

namespace eval ::tcljabberd {

    variable version 0.0.1

    variable stopped
    variable uid
    if {![info exists uid]} {
        set uid 0
    }

    variable options
    if {![info exists options]} {
        array set options {
            serveraddr {}
        }
    }

    variable URI
    array set URI {
        stream "http://etherx.jabber.org/streams"
        tls    "urn:ietf:params:xml:ns:xmpp-tls"
        sasl   "urn:ietf:params:xml:ns:xmpp-sasl"
        bind   "urn:ietf:params:xml:ns:xmpp-bind"
        sess   "urn:ietf:params:xml:ns:xmpp-session"
        err    "urn:ietf:params:xml:ns:xmpp-streams"
        client "jabber:client"
        server "jabber:server"
        db     "jabber:server:dialback"
    }

    variable log
    if {![info exists log]} {
        set log [logger::init tcljabberd]
        ${log}::setlevel debug; # warn in release
        proc ${log}::stdoutcmd {level text} {
            variable service
            puts "\[[clock format [clock seconds] -format {%H:%M:%S}]\
                $service $level\] $text"
        }
    }

}

proc ::tcljabberd::Log {level text} {
    variable log
    ${log}::${level} $text
}

proc ::tcljabberd::Start {{myaddr {}} {port 5222}} {
    variable options
    variable stopped

    if {[info exists options(socket)]} {
        return -code error "already running"
    }

    if {[string length $myaddr] > 0} {
        set options(serveraddr) $myaddr
        set myaddr "-myaddr $myaddr"
    } else {
        if {[string length $options(serveraddr)] < 1} {
            set options(serveraddr) [info hostname]
        }
    }

    set options(socket) [eval socket -server [namespace origin Accept] \
                             $myaddr $port]

    set stopped 0
    Log debug "tcljabberd service started on $options(socket)"
    return $options(socket)
}

proc ::tcljabberd::Stop {} {
    variable options
    variable stopped
    if {[info exists options(socket)]} {
        close $options(socket)
        set stopped 1
        Log debug "tcljabberd service stopped"
        unset options(socket)
    }
}

proc ::tcljabberd::Accept {channel client_addr client_port} {
    variable options
    upvar [namespace current]::state_$channel State

    catch {unset State}
    #InitState $channel
    set State(client_addr) $client_addr
    set State(client_port) $client_port
    set State(incoming)    ""
    set State(state)       accepted

    fconfigure $channel -buffering line -translation crlf -encoding utf-8
    fileevent $channel readable [list [namespace origin Service] $channel]

    Log debug "Accepted connection from $client_addr:$client_port on $channel"
    return
}

proc ::tcljabberd::Service {channel} {
    variable options
    variable uid
    upvar [namespace current]::state_$channel State
    
    if {[eof $channel]} {
        close $channel
        return
    }

    if {[catch {gets $channel line} msg]} {
        close $channel
        Log error $msg
        return
    }

    Log debug "< $line"

    switch -exact -- $State(state) {
        accepted {
            Log debug "- connecting"
            append State(incoming) $line
            if {![catch {dom parse -simple $State(incoming) doc}]} {
                set State(incoming) ""
                xmpp__init $channel $doc
            }
        }
        connected {
            Log debug "- receiving"
            append State(incoming) $line
            if {![catch {dom parse $State(incoming) doc}]} {
                set State(incoming) ""
                $doc documentElement root
                set cmd [namespace current]::xmpp_[$root nodeName]
                if {[llength [info command $cmd]] > 0} {
                    $cmd $channel $doc
                } else {
                    Log error "invalid stanza \"[$root nodeName]\""
                }
            }
        }
    }

    return
}

proc ::tcljabberd::xmpp__init {channel doc} {
    variable URI
    variable uid
    upvar [namespace current]::state_$channel State
    $doc documentElement root
    if {[string equal [$root nodeName] "stream:stream"]
        && [string equal [$root namespaceURI] $URI(stream)]} {
        
        set State(connid) [incr uid]
        set xml "<?xml version='1.0'?>\n\
<stream:stream from=\"[info hostname]\" id=\"$State(connid)\"\
   xmlns=\"$URI(client)\" xmlns:stream=\"$URI(stream)\" version=\"1.0\">\n\
   <stream:features><mechanisms xmlns=\"$URI(sasl)\">\n"
        foreach mech {PLAIN} {
            append xml "<mechanism>$mech</mechanism>\n"
        }
        append xml "</mechanisms></stream:features>"

        set State(state) connected
        puts $channel $xml
        Log debug "> $xml"
        flush $channel
    }
    return
}

proc ::tcljabberd::xmpp_presence {channel doc} {
    upvar [namespace current]::state_$channel State
    $doc documentElement root
    Log debug "presence: [$root asXML]"
    SendError $channel xml-not-well-formed ""
    return
}

proc ::tcljabberd::xmpp_iq {channel doc} {
    upvar [namespace current]::state_$channel State
    $doc documentElement root
    set nodes [$root child all]
    if {[llength $nodes] > 0} {
        foreach node $nodes {
            set action [$node nodeName]
            set actns  [$node namespaceURI]
            Log "iq $action $actns"
        }
    } else {
        Log error "iq with no childnode"
        SendError $channel xml-not-well-formed "bogus"
    }
    return
}

proc ::tcljabberd::xmpp_message {channel doc} {
    upvar [namespace current]::state_$channel State
    $doc documentElement root
    Log debug "message: [$root asXML]"
    SendError $channel xml-not-well-formed ""
    return
}

proc ::tcljabberd::SendError {channel type text} {
    variable URI
    Log debug "raise error \"$type\""
    puts $channel "<stream:error>\
         <xml-not-well-formed xmlns=\"$URI(err)\"/></stream:error>"
    flush $channel
}

# -------------------------------------------------------------------------

package provide tcljabberd $::tcljabberd::version

# -------------------------------------------------------------------------
